<?php
namespace App\Models;

use App\Core\DB;

class Visitor {
  public static function create(array $data): int {
    $pdo = DB::pdo();
    $stmt = $pdo->prepare("INSERT INTO visitors
      (company_id, visitor_name, visitor_phone, visitor_email, visitor_company, purpose, host_name, id_type, id_number, vehicle_no,
       check_in_at, status, photo_path, notes, created_by)
      VALUES (?,?,?,?,?,?,?,?,?,?,?, 'in', ?, ?, ?)");
    $stmt->execute([
      $data['company_id'],
      $data['visitor_name'],
      $data['visitor_phone'] ?? null,
      $data['visitor_email'] ?? null,
      $data['visitor_company'] ?? null,
      $data['purpose'],
      $data['host_name'] ?? null,
      $data['id_type'] ?? null,
      $data['id_number'] ?? null,
      $data['vehicle_no'] ?? null,
      $data['check_in_at'],
      $data['photo_path'] ?? null,
      $data['notes'] ?? null,
      $data['created_by'],
    ]);
    return (int)$pdo->lastInsertId();
  }

  public static function update(array $data): void {
    $pdo = DB::pdo();
    $stmt = $pdo->prepare("UPDATE visitors SET
      visitor_name=?, visitor_phone=?, visitor_email=?, visitor_company=?, purpose=?, host_name=?, id_type=?, id_number=?, vehicle_no=?,
      photo_path=?, notes=?, updated_by=?, updated_at=NOW()
      WHERE id=? AND company_id=?");
    $stmt->execute([
      $data['visitor_name'],
      $data['visitor_phone'] ?? null,
      $data['visitor_email'] ?? null,
      $data['visitor_company'] ?? null,
      $data['purpose'],
      $data['host_name'] ?? null,
      $data['id_type'] ?? null,
      $data['id_number'] ?? null,
      $data['vehicle_no'] ?? null,
      $data['photo_path'] ?? null,
      $data['notes'] ?? null,
      $data['updated_by'],
      $data['id'],
      $data['company_id'],
    ]);
  }

  public static function checkout(int $companyId, int $id, int $userId): void {
    $pdo = DB::pdo();
    $stmt = $pdo->prepare("UPDATE visitors SET status='out', check_out_at=NOW(), updated_by=? WHERE company_id=? AND id=?");
    $stmt->execute([$userId, $companyId, $id]);
  }

  public static function find(int $companyId, int $id): ?array {
    $pdo = DB::pdo();
    $stmt = $pdo->prepare("SELECT * FROM visitors WHERE company_id=? AND id=? LIMIT 1");
    $stmt->execute([$companyId, $id]);
    $row = $stmt->fetch();
    return $row ?: null;
  }

  public static function list(int $companyId, array $filters=[]): array {
    $pdo = DB::pdo();
    $sql = "SELECT * FROM visitors WHERE company_id=?";
    $params = [$companyId];

    if (!empty($filters['q'])) {
      $sql .= " AND (visitor_name LIKE ? OR visitor_phone LIKE ? OR purpose LIKE ? OR host_name LIKE ?)";
      $q = '%' . $filters['q'] . '%';
      $params = array_merge($params, [$q,$q,$q,$q]);
    }

    if (!empty($filters['status']) && in_array($filters['status'], ['in','out'], true)) {
      $sql .= " AND status=?";
      $params[] = $filters['status'];
    }

    $sql .= " ORDER BY check_in_at DESC LIMIT 500";
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    return $stmt->fetchAll();
  }

  public static function savePhotoFromBase64(string $base64Data, int $visitorId, int $companyId): ?string {
    if (empty($base64Data)) return null;
    
    // Extract base64 data
    if (preg_match('/^data:image\/(\w+);base64,/', $base64Data, $matches)) {
      $format = $matches[1];
      $data = substr($base64Data, strpos($base64Data, ',') + 1);
      $data = base64_decode($data);
    } else {
      return null;
    }

    $fileName = "visitor_" . $companyId . "_" . $visitorId . "_" . time() . "." . $format;
    $filePath = __DIR__ . '/../../storage/visitor_photos/' . $fileName;
    
    if (file_put_contents($filePath, $data) !== false) {
      return 'visitor_photos/' . $fileName;
    }
    return null;
  }

  public static function savePhotoFromUpload(array $file, int $visitorId, int $companyId): ?string {
    if (empty($file['tmp_name'])) return null;

    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($ext, ['jpg', 'jpeg', 'png', 'gif'])) return null;

    $fileName = "visitor_" . $companyId . "_" . $visitorId . "_" . time() . "." . $ext;
    $filePath = __DIR__ . '/../../storage/visitor_photos/' . $fileName;
    
    if (move_uploaded_file($file['tmp_name'], $filePath)) {
      return 'visitor_photos/' . $fileName;
    }
    return null;
  }
}
